/*
 * DamageObjectCollection.h
 *
 * Created 8/19/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/19/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef DAMAGE_OBJECT_COLLECTION_H
 #define DAMAGE_OBJECT_COLLECTION_H
 
 template <typename T> class DamageObjectCollection;
 
 #include "DamageObject.h"
 #include "pointerTo.h"
 
 #include "global.h"
 #include "StringConversion.h"
 
 #include "MapCollection.h"
 #include <map>
 
 /**
  * Class specification for DamageObjectCollection
  */
 template <typename T>
 class DamageObjectCollection : public MapCollection< std::string, PT(DamageObject<T>) >
 {
 // Data Members
 protected:
    typedef MapCollection< std::string, PT(DamageObject<T>) > MapCollection;
    typedef std::map< std::string, PT(DamageObject<T>), std::less<std::string> > DamageObjectMap;
    
 // Local Functions
 public:
    DamageObjectCollection();
    DamageObjectCollection( const DamageObjectCollection<T>& doc );
    virtual ~DamageObjectCollection();
    //inline DamageObjectCollection<T>& operator=( const DamageObjectCollection<T>& doc );
    virtual inline void add( DamageObject<T> * dmg_obj_Ptr );
    virtual inline void add( const std::string& key, DamageObject<T> * dmg_obj_Ptr );
    virtual inline void detach( const std::string& key );
    virtual inline void detach( const NodePath& node_path );
    virtual inline void detach_all();
    virtual inline DamageObject<T>* get( const std::string& key );
    virtual inline DamageObject<T>* get( const NodePath& node_path );
    virtual inline void remove( const std::string& key );
    virtual inline void remove( const NodePath& node_path );
 
 // Private Functions
 private:
    
 // Friend Functions
 public:
    
 // Public Encapsulated Classes
 public:
    /**
     *  Class specification for DamageObjectCollection<T>::iterator
     *
     *  CAVEAT:
     *  The operator->() has been overloaded to give direct access to DamageObject<T>
     *  for convenience. To access PT(DamageObject<T>), for invoking PointerTo functions, 
     *  deference the iterator using operator*() (e.g. (*itr).is_null()).
     */
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 DamageObjectCollection<T>::DamageObjectCollection()
                           : MapCollection()
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 DamageObjectCollection<T>::DamageObjectCollection( const DamageObjectCollection<T>& doc )
                           : MapCollection( doc )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 DamageObjectCollection<T>::~DamageObjectCollection()
 {
    
 }
 
 /**
  * add() adds the specified DamageObject pointer to this DamageObjectCollection.
  *
  * @param (DamageObject<T> *) dmg_obj_Ptr
  */
 template <typename T>
 inline void DamageObjectCollection<T>::add( DamageObject<T> * dmg_obj_Ptr )
 {
    DamageObjectCollection<T>::add( StringConversion::to_str( dmg_obj_Ptr->get_key() ), dmg_obj_Ptr );
 }
 
 /**
  * add() adds the specified DamageObject pointer to this DamageObjectCollection, using the specified key.
  *
  * @param (const std::string&) key
  * @param (DamageObject<T> *) dmg_obj_Ptr
  */
 template <typename T>
 inline void DamageObjectCollection<T>::add( const std::string& key, DamageObject<T> * dmg_obj_Ptr )
 {
    if ( dmg_obj_Ptr != NULL )
        MapCollection::add( key, dmg_obj_Ptr );
 }
 
 /**
  * detach() detaches the DamageObject that has the specified key, in this ObjectCollection,
  * from the DamageObject's parent.
  *
  * @param (const std::string&) key
  */
 template <typename T>
 inline void DamageObjectCollection<T>::detach( const std::string& key )
 {
    PT(DamageObject<T>) dmg_obj_Ptr( MapCollection::get( key ) );
    
    nassertv( dmg_obj_Ptr != NULL );
    
    dmg_obj_Ptr->detach_node();
 }
 
 /**
  * detach() detaches the DamageObject matching the specified node_path,
  * in this DamageObjectCollection, from its parent. This only works if 
  * the DamageObject was stored using the DamageObject's key.
  *
  * @param (const NodePath&) node_path
  */
 template <typename T>
 inline void DamageObjectCollection<T>::detach( const NodePath& node_path )
 {
    DamageObjectCollection<T>::detach( StringConversion::to_str( node_path.get_key() ) );
 }
 
 /**
  * detach_all() detach all DamageObjects, in this DamageObjectCollection, 
  * from their parents.
  */
 template <typename T>
 inline void DamageObjectCollection<T>::detach_all()
 {
    MapCollection::iterator dmg_obj_Itr( MapCollection::begin() );
    MapCollection::iterator end_Itr( MapCollection::end() );
    
    while ( dmg_obj_Itr != end_Itr )
    {
        dmg_obj_Itr->detach_node();
        ++dmg_obj_Itr;
    }
 }
 
 /**
  * get() returns a pointer to the DamageObject that has the specified key.
  *
  * @param (const std::string&) key
  * @return DamageObject<T>*
  */
 template <typename T>
 inline DamageObject<T>* DamageObjectCollection<T>::get( const std::string& key )
 {
    //return MapCollection::get( key );
    MapCollection::iterator dmg_obj_Itr( MapCollection::find( key ) );
    
    // if the DamageObject is not found in DamageObjectCollection
    if ( dmg_obj_Itr == MapCollection::end() )
        return NULL;
    else // the DamageObject matching the key is found in the DamageObjectCollection
        return *dmg_obj_Itr;
 }
 
 /**
  * get() returns a pointer to the DamageObject matching the specified node_path.
  * If the specified node_path is not a DamageObject contained within this collection,
  * then NULL is returned. This only works if the DamageObject was stored using the 
  * DamageObject's key.
  *
  * @param (const NodePath&) node_path
  * @return DamageObject<T>*
  */
 template <typename T>
 inline DamageObject<T>* DamageObjectCollection<T>::get( const NodePath& node_path )
 {
    return DamageObjectCollection<T>::get( StringConversion::to_str( node_path.get_key() ) );
 }
 
 /**
  * remove() removes the DamageObject that has the specified key from this ObjectCollection.
  *
  * @param (const std::string&) key
  */
 template <typename T>
 inline void DamageObjectCollection<T>::remove( const std::string& key )
 {
    MapCollection::remove( key );
 }
 
 /**
  * remove() removes the DamageObject matching the specified node_path
  * in this DamageObjectCollection. This only works if the DamageObject was 
  * stored using the DamageObject's key.
  *
  * @param (const NodePath&) node_path
  */
 template <typename T>
 inline void DamageObjectCollection<T>::remove( const NodePath& node_path )
 {
    DamageObjectCollection<T>::remove( StringConversion::to_str( node_path.get_key() ) );
 }
 
 /** FRIEND FUNCTIONS **/
 
 #endif // DAMAGE_OBJECT_COLLECTION_H